/**
 * @file Matrix.cpp
 * @brief sNX`
 * @author cherub
 * @date 2005.09.16
 */
#include "Matrix.h"

#include <GSL/gsl_math.h>
#include <GSL/gsl_blas.h>
#include <GSL/gsl_linalg.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#endif

/**
 * @fn CMatrix::CMatrix(void)
 * @brief ftHgRXgN^
 */
CMatrix::CMatrix(void)
{
	m_pmMatrix = NULL;
}

/**
 *	@fn CMatrix::CMatrix(size_t unRow, size_t unCol)
 *	@brief TCYwRXgN^
 *	@param unRow [in] s
 *	@param unCol [in] 
 */
CMatrix::CMatrix(size_t unRow, size_t unCol)
{
	m_pmMatrix = gsl_matrix_calloc(unRow, unCol);
}

/**
 * @fn CMatrix::CMatrix(size_t unRow, size_t unCol, double dInit)
 * @brief qTCYwRXgN^
 * @param unRow [in] s
 * @param unCol [in] 
 * @param dInit [in] q
 */
CMatrix::CMatrix(size_t unRow, size_t unCol, double dInit)
{
	m_pmMatrix = NULL;
	Initialize(unRow, unCol, dInit);
}

/**
 * @fn CMatrix::CMatrix(const CMatrix &m)
 * @brief Rs[RXgN^
 * @param m [in] Rs[s
 */
CMatrix::CMatrix(const CMatrix &m)
{
	m_pmMatrix = gsl_matrix_alloc(m.m_pmMatrix->size1, m.m_pmMatrix->size2);
	gsl_matrix_memcpy(m_pmMatrix, m.m_pmMatrix);
}

/**
 * @fn CMatrix::~CMatrix(void)
 * @brief fXgN^
 */
CMatrix::~CMatrix(void)
{
	Destroy();
}

/**
 * @fn CMatrix& CMatrix::Initialize(size_t unRow, size_t unCol, double dInit)
 * @brief ֐
 * @param unRow [in] s
 * @param unCol [in] 
 * @param dInit [in] qDftHg0
 * @return gւ̎Q
 */
CMatrix& CMatrix::Initialize(size_t unRow, size_t unCol, double dInit)
{
	Destroy();

	m_pmMatrix = gsl_matrix_calloc(unRow, unCol);
	gsl_matrix_set_all(m_pmMatrix, dInit);

	return *this;
}

/**
 * @fn CMatrix& CMatrix::ZeroClear(void)
 * @brief s̃[NA
 * @return gւ̎Q
 */
CMatrix& CMatrix::ZeroClear(void)
{
	gsl_matrix_set_zero(m_pmMatrix);
	return *this;
}

/**
 * @fn CMatrix& CMatrix::Clear(double dInit)
 * @brief CӒlɂsNA
 * @return gւ̎Q
 */
CMatrix& CMatrix::Clear(double dInit)
{
	gsl_matrix_set_all(m_pmMatrix, dInit);
	return *this;
}

/**
 * @fn void CMatrix::Destroy(void)
 * @brief IuWFNg̔j
 * @return Ȃ
 */
void CMatrix::Destroy(void)
{
	if (m_pmMatrix != NULL)
		gsl_matrix_free(m_pmMatrix);
	m_pmMatrix = NULL;
}

/**
 * @fn CVector CMatrix::RowVector(size_t r) const
 * @brief sxNg̎擾
 * @param r [in] sCfNX
 * @return sxNg
 */
CVector CMatrix::RowVector(size_t r) const
{
	CVector v(m_pmMatrix->size2);

	for (size_t j = 0;j < v.Size();j++)
		v[j] = Data(r, j);
	return v;
}

/**
 * @fn CVector CMatrix::ColVector(size_t c) const
 * @brief xNg̎擾
 * @param c [in] CfNX
 * @return xNg
 */
CVector CMatrix::ColVector(size_t c) const
{
	CVector v(m_pmMatrix->size1);
	
	for (size_t i = 0;i < v.Size();i++)
		v[i] = Data(i, c);
	return v;
}

/**
 * @fn 	CMatrix& CMatrix::operator = (const CMatrix &m)
 * @brief Zq
 * @param m [in] s
 * @return gւ̎Q
 */
CMatrix& CMatrix::operator = (const CMatrix &m)
{
	Destroy();

	m_pmMatrix = gsl_matrix_alloc(m.m_pmMatrix->size1, m.m_pmMatrix->size2);
	gsl_matrix_memcpy(m_pmMatrix, m.m_pmMatrix);
	return *this;
}

/* sv */
/**
 * @fn CMatrix CMatrix::operator - () const
 * @brief s̑Svf̕𔽓]
 * @return Z
 */
/*
CMatrix CMatrix::operator - () const
{
	CMatrix tmp(*this);
	gsl_matrix_scale(tmp.m_pmMatrix, -1.0);
	return tmp;
}
*/

/**
 * @fn bool CMatrix::operator == (const CMatrix &m) const
 * @brief s̈v
 * @retval true s񂪈v
 * @retval false s񂪕sv
 */
bool CMatrix::operator == (const CMatrix &m) const
{
	for (size_t r = 0;r < m_pmMatrix->size1;r++)
	for (size_t c = 0;c < m_pmMatrix->size2;c++)
	{
		if (gsl_matrix_get(m_pmMatrix, r, c) != gsl_matrix_get(m.m_pmMatrix, r, c))
			return false;
	}
	return true;
}

/*  */
/**
 * @fn CMatrix CMatrix::operator + (const CMatrix &m) const
 * @brief ZZq
 * @param m [in] ZΏ
 * @return Z
 */
CMatrix CMatrix::operator + (const CMatrix &m) const
{
	//CMatrix tmp(m_pmMatrix->size1, m_pmMatrix->size2);
	if ((m_pmMatrix->size1 != m.m_pmMatrix->size1)
		|| (m_pmMatrix->size2 != m.m_pmMatrix->size2))
	{
		return *this;
	}
	CMatrix tmp = *this;
	gsl_matrix_add(tmp.m_pmMatrix, m.m_pmMatrix);
	return tmp;
}

/*  */
/**
 * @fn CMatrix CMatrix::operator - (const CMatrix &m) const
 * @brief ZZq
 * @param m [in] ZΏ
 * @return Z
 */
CMatrix CMatrix::operator - (const CMatrix &m) const
{
	if ((m_pmMatrix->size1 != m.m_pmMatrix->size1)
		|| (m_pmMatrix->size2 != m.m_pmMatrix->size2))
	{
		return *this;
	}
	//CMatrix tmp(m_pmMatrix->size1, m_pmMatrix->size2);
	CMatrix tmp = *this;
	gsl_matrix_sub(tmp.m_pmMatrix, m.m_pmMatrix);
	return tmp;
}

/**
 * @fn CMatrix CMatrix::operator * (const CMatrix &m) const
 * @brief ZZq
 * @param m [in] ZΏ
 * @return Z
 */
CMatrix CMatrix::operator * (const CMatrix &m) const
{
	CMatrix tmp(m_pmMatrix->size1, m.m_pmMatrix->size2);
	//
	// gsl_blas_dgemm(OpA, OpB, alpha, matA, matB, beta, matC)
	//  matC = alpha * OpA(matA) * OpB(matB) + beta * matC;
	//
	gsl_blas_dgemm(CblasNoTrans, CblasNoTrans, 1.0, m_pmMatrix, m.m_pmMatrix, 0.0, tmp.m_pmMatrix);
	return tmp;
}

/**
 * @fn CMatrix CMatrix::operator + (double d) const
 * @brief XJZZq
 * @param d [in] ZΏ
 * @return Z
 */
CMatrix CMatrix::operator + (double d) const
{
	CMatrix tmp = *this;
	gsl_matrix_add_constant(tmp.m_pmMatrix, d);
	return tmp;
}

/**
 * @fn CMatrix CMatrix::operator * (double d) const
 * @brief XJZZq
 * @param d [in] ZΏ
 * @return Z
 */
CMatrix CMatrix::operator * (double d) const
{
	CMatrix tmp = *this;
	gsl_matrix_scale(tmp.m_pmMatrix, d);
	return tmp;
}

/**
 * @fn CMatrix CMatrix::operator += (const CMatrix &m)
 * @brief ZZq
 * @param m [in] ZΏ
 * @return gւ̎Q
 */
CMatrix& CMatrix::operator +=(const CMatrix &m)
{
	gsl_matrix_add(m_pmMatrix, m.m_pmMatrix);
	return *this;
}

/**
 * @fn CMatrix CMatrix::operator -= (const CMatrix &m)
 * @brief ZZq
 * @param m [in] ZΏ
 * @return gւ̎Q
 */
CMatrix& CMatrix::operator -= (const CMatrix &m)
{
	gsl_matrix_sub(m_pmMatrix, m.m_pmMatrix);
	return *this;
}

/**
 * @fn CMatrix CMatrix::operator *= (const CMatrix &m)
 * @brief ZZq
 * @param m [in] ZΏ
 * @return gւ̎Q
 */
CMatrix& CMatrix::operator *= (const CMatrix &m)
{
	CMatrix tmp(m_pmMatrix->size1, m.m_pmMatrix->size2);
	gsl_blas_dgemm(CblasNoTrans, CblasNoTrans, 1.0, tmp.m_pmMatrix, m.m_pmMatrix, 0.0, m_pmMatrix);
	return *this;
}

/**
 * @fn CMatrix CMatrix::operator += (double d)
 * @brief XJZZq
 * @param d [in] ZΏ
 * @return gւ̎Q
 */
CMatrix& CMatrix::operator += (double d)
{
	gsl_matrix_add_constant(m_pmMatrix, d);
	return *this;
}

/**
 * @fn CMatrix CMatrix::operator *= (double d)
 * @brief XJZZq
 * @param d [in] ZΏ
 * @return gւ̎Q
 */
CMatrix& CMatrix::operator *= (double d)
{
	gsl_matrix_scale(m_pmMatrix, d);
	return *this;
}

/**
 * @fn CVector CMatrix::operator * (const CVector &v) const
 * @brief xNgƂ̏ZZq
 * @param v [in] ZΏ
 * @return Z
 */
CVector CMatrix::operator * (const CVector &v) const
{
	CVector vTmp(m_pmMatrix->size1);
	//
	// gsl_blas_dgemv(Op, alpha, matA, vecX, beta, vecY)
	//  vecY = alpha * Op(A) * vecX + beta * vecY;
	//
	gsl_blas_dgemv(CblasNoTrans, 1.0, m_pmMatrix, v.m_pvVector, 0.0f, vTmp.m_pvVector);
	return vTmp;
}

/**
 * @fn CMatrix& CMatrix::LoadIdentity(size_t n)
 * @brief n̒Pʍs𐶐
 * @param n [in] 
 * @return Z
 */
CMatrix& CMatrix::LoadIdentity(size_t n)
{
	if (m_pmMatrix != NULL)
		gsl_matrix_free(m_pmMatrix);
	m_pmMatrix = gsl_matrix_alloc(n, n);
	gsl_matrix_set_identity(m_pmMatrix);

	return *this;
}

/**
 * @fn CMatrix CMatrix::Transpose(void) const
 * @brief ]ušvZ
 * @return Z
 */
CMatrix CMatrix::Transpose(void) const
{
	CMatrix tmp(m_pmMatrix->size2, m_pmMatrix->size1);
	gsl_matrix_transpose_memcpy(tmp.m_pmMatrix, m_pmMatrix);
	return tmp;
}

/**
 * @fn CMatrix CMatrix::InverseLUD(void) const
 * @brief LUoR̋tsvZ(ŝ)
 * @return Z
 */
CMatrix CMatrix::InverseLUD(void) const
{
	// s񂩂̃`FbN
	if (m_pmMatrix->size1 != m_pmMatrix->size2)
		return *this;

	CMatrix inv(m_pmMatrix->size1, m_pmMatrix->size1);
	CMatrix tmp = *this;
	int signum;
	gsl_permutation *p = gsl_permutation_alloc(m_pmMatrix->size1);

	gsl_linalg_LU_decomp(tmp.m_pmMatrix, p, &signum);
	gsl_linalg_LU_invert(tmp.m_pmMatrix, p, inv.m_pmMatrix);

	gsl_permutation_free(p);
	return inv;
}

/**
 * @fn CMatrix CMatrix::Determinant(void) const
 * @brief LUoR̍s񎮂̌vZ
 * @return Z
 */
double CMatrix::Determinant(void) const
{
	// s񂩂̃`FbN
	if (m_pmMatrix->size1 != m_pmMatrix->size2)
		return 0.0;

	CMatrix tmp = *this;
	int signum;
	gsl_permutation *p = gsl_permutation_alloc(m_pmMatrix->size1);

	gsl_linalg_LU_decomp(tmp.m_pmMatrix, p, &signum);
	double det = gsl_linalg_LU_det(tmp.m_pmMatrix, signum);

	gsl_permutation_free(p);
	return det;
}

/**
 * @fn CMatrix CMatrix::Trace(void) const
 * @brief s̑Ίpvf̘a
 * @return Z
 */
double CMatrix::Trace(void) const
{
	int i;
	double trace = 0;
	CMatrix tmp = *this;

	// s񂩂̃`FbN
	if (m_pmMatrix->size1 != m_pmMatrix->size2)
		return 0.0;

	for(i = 0; i < m_pmMatrix->size1; i++)
	{
		trace += tmp(i, i);
	}
	
	return(trace);
}

/**
 * @fn CMatrix CMatrix::InverseSVD(double dThreshold) const
 * @brief ْloR̋[tsvZ
 * @param dThreshold [in] tsvZɗpْl臒l
 * @return Z
 */
CMatrix CMatrix::InverseSVD(double dThreshold) const
{
	CMatrix mU;
	if (m_pmMatrix->size1 >= m_pmMatrix->size2)
		mU = *this;
	else
		mU = Transpose();

	CMatrix mInv(mU.m_pmMatrix->size2, mU.m_pmMatrix->size1);
	CMatrix mV(mU.m_pmMatrix->size2, mU.m_pmMatrix->size2);
	gsl_vector *vS = gsl_vector_alloc(mU.m_pmMatrix->size2);
	gsl_vector *vW = gsl_vector_alloc(mU.m_pmMatrix->size2);

	CMatrix mS(vS->size, vS->size);

	gsl_linalg_SV_decomp(mU.m_pmMatrix, mV.m_pmMatrix, vS, vW);

	for (size_t r = 0;r < vS->size;r++)
	{
		if (gsl_vector_get(vS, r) < dThreshold)
			mS(r, r) = 0.0;
		else
			mS(r, r) = 1.0 / gsl_vector_get(vS, r);
	}

	gsl_vector_free(vS);
	gsl_vector_free(vW);

	mU = mV * mS * mU.Transpose();
	if (m_pmMatrix->size1 < m_pmMatrix->size2)
		return mU.Transpose();
	return mU;
}

/**
 * @fn CMatrix CMatrix::PseudoInverse(void) const
 * @brief [tsvZ
 * @return Z
 */
CMatrix CMatrix::PseudoInverse(void) const
{
	if (m_pmMatrix->size1 < m_pmMatrix->size2)
		return PseudoInverseRight();
	if (m_pmMatrix->size1 > m_pmMatrix->size2)
		return PseudoInverseLeft();

	return InverseLUD();
}

/**
 * @fn CMatrix CMatrix::PseudoInverseLeft(void) const
 * @brief (A^+)(A)𖞂[tšvZ
 * @return Z
 */
CMatrix CMatrix::PseudoInverseLeft(void) const
{
	return (Transpose() * *this).InverseLUD() * Transpose();
}

/**
 * @fn CMatrix CMatrix::PseudoInverseRight(void) const
 * @brief (A)(A^+)𖞂[tšvZ
 * @return Z
 */
CMatrix CMatrix::PseudoInverseRight(void) const
{
	return Transpose() * (*this * Transpose()).InverseLUD();
}

/**
 * @fn double CMatrix::MinElement(void) const
 * @brief ŏl̎擾
 * @return s񒆂̍ŏl
 */
double CMatrix::MinElement(void) const
{
	return gsl_matrix_min(m_pmMatrix);
}

/**
 * @fn double CMatrix::MaxElement(void) const
 * @brief ől̎擾
 * @return s񒆂̍ől
 */
double CMatrix::MaxElement(void) const
{
	return gsl_matrix_max(m_pmMatrix);
}

/**
 * @fn double CMatrix::MinMaxElement(double &dMin, double &dMax) const
 * @brief ŏlEől̎擾
 * @param dMin [out] s񒆂̍ŏl
 * @param dMax [out] s񒆂̍ől
 * @return Ȃ
 */
void CMatrix::MinMaxElement(double &dMin, double &dMax) const
{
	gsl_matrix_minmax(m_pmMatrix, &dMin, &dMax);
}

/**
 * @fn void CMatrix::MinIndex(size_t &r, size_t &c) const
 * @brief ŏlւ̃CfNX̎擾
 * @param r s񒆂̍ŏl̍sCfNX
 * @param c s񒆂̍ŏl̗CfNX
 * @return Ȃ
 */
void CMatrix::MinIndex(size_t &r, size_t &c) const
{
	size_t i, j;
	gsl_matrix_min_index(m_pmMatrix, &i, &j);
	r = i;
	c = j;
}

/**
 * @fn void CMatrix::MaxIndex(size_t &r, size_t &c) const
 * @brief őlւ̃CfNX̎擾
 * @param r s񒆂̍ől̍sCfNX
 * @param c s񒆂̍ől̗CfNX
 * @return Ȃ
 */
void CMatrix::MaxIndex(size_t &r, size_t &c) const
{
	size_t i, j;
	gsl_matrix_max_index(m_pmMatrix, &i, &j);
	r = i;
	c = j;
}

/**
 * @fn void CMatrix::MinMaxIndex(size_t &min_r, size_t &min_c, size_t &max_r, size_t &max_c) const
 * @brief ŏlEőlCfNX̎擾
 * @param min_r [out] s񒆂̍ŏl̍sԍ
 * @param min_c [out] s񒆂̍ŏl̗ԍ
 * @param max_r [out] s񒆂̍ől̍sԍ
 * @param max_c [out] s񒆂̍ől̗ԍ
 * @return Ȃ
 */
void CMatrix::MinMaxIndex(size_t &min_r, size_t &min_c, size_t &max_r, size_t &max_c) const
{
	size_t mini, minj, maxi, maxj;
	gsl_matrix_minmax_index(m_pmMatrix, &mini, &minj, &maxi, &maxj);
	min_r = mini;
	min_c = minj;
	max_r = maxi;
	max_c = maxj;
}
